<?php
// components/location-slider.php

// Get the slides from ACF with simple, safe approach
$location_slides = array();

if (function_exists('get_field')) {
    $slides_data = get_field('location_slides');
    
    // Debug: Check what we're getting from ACF
    error_log('ACF Slides Data: ' . print_r($slides_data, true));
    
    if ($slides_data && is_array($slides_data)) {
        foreach ($slides_data as $slide) {
            // Simple, safe way to get data
            $city_name = '';
            $bg_url = '';
            
            // Get city name - try different possible field structures
            if (isset($slide['city_name']) && is_string($slide['city_name'])) {
                $city_name = $slide['city_name'];
            }
            
            // Get background image - try different possible field structures
            if (isset($slide['background_image']) && is_array($slide['background_image']) && isset($slide['background_image']['url'])) {
                $bg_url = $slide['background_image']['url'];
            } elseif (isset($slide['background_image']) && is_numeric($slide['background_image'])) {
                $bg_url = wp_get_attachment_url($slide['background_image']);
            } elseif (isset($slide['background_image']) && is_string($slide['background_image'])) {
                $bg_url = $slide['background_image'];
            }
            
            // Only add if we have both values
            if ($city_name && $bg_url) {
                $location_slides[] = array(
                    'city_name' => $city_name,
                    'bg_url' => $bg_url
                );
            }
        }
    }
}

// If no valid slides found, use demo data
if (empty($location_slides)) {
    $location_slides = array(
        array(
            'city_name' => 'Amsterdam',
            'bg_url' => 'https://s3-us-west-2.amazonaws.com/s.cdpn.io/537051/city--1-min-min.jpg'
        ),
        array(
            'city_name' => 'Rome', 
            'bg_url' => 'https://s3-us-west-2.amazonaws.com/s.cdpn.io/537051/city--2-min-min.jpg'
        ),
        array(
            'city_name' => 'New York',
            'bg_url' => 'https://s3-us-west-2.amazonaws.com/s.cdpn.io/537051/city--3-min-min.jpg'
        )
    );
}

// Count the actual slides we'll use
$slide_count = count($location_slides);
error_log("Final Slide Count: " . $slide_count);
?>

<section class="location-slider-section">
    <div class="cont">
        <div class="slider">
            <?php foreach ($location_slides as $index => $slide): 
                $slide_number = $index + 1;
                $city_name = $slide['city_name'];
                $bg_url = $slide['bg_url'];
                $first_letter = substr($city_name, 0, 1);
                
                // Split city name into chunks
                $chunk_size = max(1, ceil(strlen($city_name) / 4));
                $chunks = str_split($city_name, $chunk_size);
                ?>
                
                <div data-target="<?php echo $slide_number; ?>" class="slide slide--<?php echo $slide_number; ?>">
                    <div class="slide__darkbg slide--<?php echo $slide_number; ?>__darkbg" 
                         style="background-image: url('<?php echo esc_url($bg_url); ?>')"></div>
                    <div class="slide__text-wrapper slide--<?php echo $slide_number; ?>__text-wrapper">
                        <div class="slide__letter slide--<?php echo $slide_number; ?>__letter" 
                             style="background-image: url('<?php echo esc_url($bg_url); ?>')">
                            <?php echo esc_html($first_letter); ?>
                        </div>
                        <?php foreach ($chunks as $chunk_index => $chunk): ?>
                            <div class="slide__text slide__text--<?php echo $chunk_index + 1; ?>">
                                <?php echo esc_html($chunk); ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
        
        <ul class="nav">
            <?php foreach ($location_slides as $index => $slide): 
                $slide_number = $index + 1;
                ?>
                <li data-target="<?php echo $slide_number; ?>" class="nav__slide nav__slide--<?php echo $slide_number; ?> <?php echo $index === 0 ? 'nav-active' : ''; ?>"></li>
            <?php endforeach; ?>
        </ul>
        
        <div data-target='right' class="side-nav side-nav--right"></div>
        <div data-target='left' class="side-nav side-nav--left"></div>
    </div>
</style>

<style>
    :root {
        --bgClr: rgba(11, 15, 39, 0.83);
        --txtClr: #fff;
        --animSpd: 750ms;
    }

    .location-slider-section {
        width: 100%;
        overflow: hidden;
    }

    .cont {
        position: relative;
        height: 100vh;
        overflow: hidden;
        border-bottom-right-radius: 15vw;
        border-top-right-radius: 15vw;
    }

    .slider {
        position: relative;
        height: 100%;
        transform: translate3d(0, 0, 0);
        will-change: transform;
        cursor: grab;
        user-select: none;
        width: <?php echo $slide_count * 100; ?>%;
    }

    .slider.animation {
        transition: transform var(--animSpd) ease-in-out;
    }

    .slider.animation .slide__darkbg,
    .slider.animation .slide__text,
    .slider.animation .slide__letter {
        transition: transform var(--animSpd) ease-in-out;
    }

    .slide {
        position: absolute;
        top: 0;
        width: <?php echo 100 / $slide_count; ?>%;
        height: 100%;
        overflow: hidden;
    }

    <?php foreach ($location_slides as $index => $slide): 
        $slide_number = $index + 1;
        $position = ($slide_number - 1) * (100 / $slide_count);
    ?>
        .slide--<?php echo $slide_number; ?> {
            left: <?php echo $position; ?>%;
        }
    <?php endforeach; ?>

    .slide__darkbg {
        position: absolute;
        width: 100%;
        height: 100%;
        top: 0;
        left: 0;
        transform: translate3d(0, 0, 0);
        will-change: transform;
        z-index: 10;
        background-size: cover !important;
        background-position: center center !important;
        background-repeat: no-repeat !important;
    }

    <?php foreach ($location_slides as $index => $slide): 
        $slide_number = $index + 1;
        $bg_position = ($slide_number - 1) * 50;
    ?>
        .slide--<?php echo $slide_number; ?>__darkbg {
            left: -<?php echo $bg_position; ?>%;
            transform: translate3d(0, 0, 0);
            will-change: transform;
        }
        
        .slide--<?php echo $slide_number; ?>__darkbg:after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: var(--bgClr);
        }
    <?php endforeach; ?>

    .slide__text-wrapper {
        position: absolute;
        display: flex;
        justify-content: center;
        align-items: center;
        width: 100%;
        height: 100%;
        z-index: 15;
    }

    .slide__letter {
        position: absolute;
        display: flex;
        align-items: center;
        justify-content: center;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        transform: translate3d(0, 0, 0);
        -webkit-text-fill-color: transparent;
        -webkit-background-clip: text;
        background-clip: text;
        font-size: 50vw;
        font-weight: 800;
        color: #000;
        z-index: 2;
        will-change: transform;
        user-select: none;
        background-size: cover !important;
        background-position: center center !important;
        background-repeat: no-repeat !important;
    }

    .slide__text {
        font-size: 8vw;
        font-weight: 800;
        text-transform: uppercase;
        transform: translate3d(0, 0, 0);
        letter-spacing: 12px;
        color: var(--txtClr);
        will-change: transform;
        user-select: none;
    }

    .slide__text:nth-child(odd) {
        z-index: 2;
    }

    .slide__text:nth-child(even) {
        z-index: 1;
    }

    .nav {
        position: absolute;
        bottom: 25px;
        left: 50%;
        transform: translateX(-50%);
        list-style-type: none;
        z-index: 10;
        margin: 0;
        padding: 0;
        display: flex;
        gap: 10px;
    }

    .nav__slide {
        position: relative;
        display: inline-block;
        width: 28px;
        height: 28px;
        border-radius: 50%;
        border: 2px solid #fff;
        cursor: pointer;
    }

    @media screen and (max-width: 400px) {
        .nav__slide {
            width: 22px;
            height: 22px;
        }
    }

    .nav__slide:hover:after {
        transform: translate(-50%, -50%) scale(1, 1);
        opacity: 1;
    }

    .nav__slide:after {
        content: '';
        position: absolute;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%) scale(0, 0);
        width: 75%;
        height: 75%;
        border-radius: 50%;
        background-color: #fff;
        opacity: 0;
        transition: 300ms;
    }

    .nav-active:after {
        transform: translate(-50%, -50%) scale(1, 1);
        opacity: 1;
    }

    .side-nav {
        position: absolute;
        width: 10%;
        height: 100%;
        top: 0;
        z-index: 20;
        cursor: pointer;
        opacity: 0;
        transition: 300ms;
    }

    .side-nav:hover {
        opacity: .1;
    }

    .side-nav--right {
        right: 0;
        background-image: linear-gradient(to right, rgba(255, 255, 255, 0) 0%, rgb(238, 215, 255) 100%);
    }

    .side-nav--left {
        left: 0;
        background-image: linear-gradient(to left, rgba(255, 255, 255, 0) 0%, rgb(238, 215, 255) 100%);
    }
</style>

<script>
jQuery(document).ready(function($) {
    'use strict';
    
    const $slider = $('.slider');
    const $nav = $('.nav');
    const winW = $(window).width();
    const animSpd = 750;
    const distOfLetGo = winW * 0.2;
    let curSlide = 1;
    let animation = false;
    let diff = 0;
    
    // Get number of slides dynamically
    const numOfSlides = <?php echo $slide_count; ?>;
    
    console.log('Slider initialized with', numOfSlides, 'slides');
    
    // Navigation functions
    function bullets(dir) {
        $('.nav__slide').removeClass('nav-active');
        $('.nav__slide--' + dir).addClass('nav-active');
    }
    
    function timeout() {
        animation = false;
    }
    
    function pagination(direction) {
        animation = true;
        diff = 0;
        $slider.addClass('animation');
        
        const slideWidth = 100 / numOfSlides;
        $slider.css({
            'transform': 'translate3d(-' + ((curSlide - direction) * slideWidth) + '%, 0, 0)'
        });
        
        $('.slide__darkbg').css({
            'transform': 'translate3d(' + ((curSlide - direction) * 50) + '%, 0, 0)'
        });
        
        $('.slide__letter, .slide__text').css({
            'transform': 'translate3d(0, 0, 0)'
        });
        
        setTimeout(timeout, animSpd);
    }
    
    function navigateRight() {
        if (animation) return;
        
        if (curSlide >= numOfSlides) {
            curSlide = 1;
        } else {
            curSlide++;
        }
        
        pagination(1);
        bullets(curSlide);
    }
    
    function navigateLeft() {
        if (animation) return;
        
        if (curSlide <= 1) {
            curSlide = numOfSlides;
        } else {
            curSlide--;
        }
        
        pagination(1);
        bullets(curSlide);
    }
    
    function toDefault() {
        pagination(1);
    }
    
    // Event handlers
    $(document).on('mousedown touchstart', '.slide', function(e) {
        if (animation) return;
        
        let startX = e.pageX || e.originalEvent.touches[0].pageX;
        $slider.removeClass('animation');
        
        $(document).on('mousemove touchmove', function(e) {
            let x = e.pageX || e.originalEvent.touches[0].pageX;
            diff = startX - x;
            
            // Prevent sliding beyond first/last slide
            if ((curSlide === 1 && diff < 0) || (curSlide === numOfSlides && diff > 0)) {
                diff = 0;
                return;
            }
            
            const slideWidth = 100 / numOfSlides;
            $slider.css({
                'transform': 'translate3d(-' + ((curSlide - 1) * slideWidth + (diff / 30)) + '%, 0, 0)'
            });
            
            $('.slide__darkbg').css({
                'transform': 'translate3d(' + ((curSlide - 1) * 50 + (diff / 60)) + '%, 0, 0)'
            });
            
            $('.slide__letter').css({
                'transform': 'translate3d(' + (diff / 60) + 'vw, 0, 0)'
            });
            
            $('.slide__text').css({
                'transform': 'translate3d(' + (diff / 15) + 'px, 0, 0)'
            });
        });
    });
    
    $(document).on('mouseup touchend', function() {
        $(document).off('mousemove touchmove');
        
        if (animation) return;
        
        if (diff >= distOfLetGo) {
            navigateRight();
        } else if (diff <= -distOfLetGo) {
            navigateLeft();
        } else {
            toDefault();
        }
    });
    
    $(document).on('click', '.nav__slide:not(.nav-active)', function() {
        let target = +$(this).data('target');
        bullets(target);
        curSlide = target;
        pagination(1);
    });
    
    $(document).on('click', '.side-nav', function() {
        let target = $(this).data('target');
        
        if (target === 'right') navigateRight();
        if (target === 'left') navigateLeft();
    });
    
    $(document).on('keydown', function(e) {
        if (e.which === 39) navigateRight();
        if (e.which === 37) navigateLeft();
    });
    
    // Handle window resize
    $(window).on('resize', function() {
        const slideWidth = 100 / numOfSlides;
        $slider.css('transform', 'translate3d(-' + ((curSlide - 1) * slideWidth) + '%, 0, 0)');
        $('.slide__darkbg').css('transform', 'translate3d(' + ((curSlide - 1) * 50) + '%, 0, 0)');
    });
});
</script>