<?php
/**
 * Template Name: contact
 * Created on : Aug 18, 2025
 * Author: Minidu Sathsara
 */
get_header(); ?>

<?php get_template_part('components/canvas'); ?>

<style>
.contact-page-wrapper {
    min-height: 100vh;
    display: flex;
    align-items: center;
    justify-content: center;
    padding: 40px 20px;
    position: relative;
    z-index: 1;
}

.contact-container {
    width: 100%;
    max-width: 700px;
    margin: 0 auto;
    position: relative;
    z-index: 2;
}

.contact-content {
   background: rgba(255, 255, 255, 0.15);
    backdrop-filter: blur(20px);
    border-radius: 20px;
    padding: 40px;
    box-shadow: 0 15px 35px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.3);
}

.contact-title {
    font-size: 2.5rem;
    margin-bottom: 15px;
    text-align: center;
    color: #2d3748;
    font-weight: 700;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.contact-description {
    text-align: center;
    margin-bottom: 40px;
    color: #4a5568;
    font-size: 1.1rem;
    line-height: 1.6;
}

.modern-contact-form {
    display: flex;
    flex-direction: column;
    gap: 25px;
}

.form-row {
    display: flex;
    gap: 20px;
}

.form-group {
    display: flex;
    flex-direction: column;
    flex: 1;
}

.form-group label {
    margin-bottom: 8px;
    font-weight: 600;
    color: #ffffffff;
    font-size: 0.95rem;
}

.form-group input, .form-group textarea {
    padding: 14px 16px;
    border: 2px solid rgba(255, 255, 255, 0.15);
    border-radius: 12px;
    background: rgba(252, 129, 129, 0.05);
    font-size: 1rem;
    transition: all 0.3s ease;
    color: #2d3748;
}

.form-group input:focus,
.form-group textarea:focus {
    outline: none;
    border-color: #667eea;
    background: rgba(255, 255, 255, 1);
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    transform: translateY(-1px);
}

.form-group input::placeholder,
.form-group textarea::placeholder {
    color: #a0aec0;
}

.form-group textarea {
    resize: vertical;
    min-height: 120px;
}

.checkbox-group {
    flex-direction: row;
    align-items: center;
    gap: 10px;
}

.checkbox-group input {
    width: 18px;
    height: 18px;
    accent-color: #667eea;
}

.checkbox-group label {
    margin-bottom: 0;
    font-weight: normal;
    color: #4a5568;
}

.submit-btn {
    background: #e50914;
    color: white;
    border: none;
    padding: 16px 30px;
    border-radius: 12px;
    font-size: 1.1rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    margin-top: 10px;
    position: relative;
    overflow: hidden;
}

.submit-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 12px 25px rgba(102, 126, 234, 0.4);
}

.submit-btn:active {
    transform: translateY(0);
}

.submit-btn::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
    transition: left 0.5s;
}

.submit-btn:hover::before {
    left: 100%;
}

/* Form validation styles */
.form-group input:invalid:not(:focus):not(:placeholder-shown),
.form-group textarea:invalid:not(:focus):not(:placeholder-shown) {
    border-color: #ffffffff;
    background: rgba(252, 129, 129, 0.05);
}

.form-group input:valid:not(:focus):not(:placeholder-shown),
.form-group textarea:valid:not(:focus):not(:placeholder-shown) {
    border-color: #ffffffff;
}

/* Responsive design */
@media (max-width: 768px) {
    .contact-content {
        padding: 30px 25px;
    }
    
    .form-row {
        flex-direction: column;
        gap: 25px;
    }
    
    .contact-title {
        font-size: 2rem;
    }
    
    .contact-page-wrapper {
        padding: 20px 15px;
    }
}

/* Loading state for form submission */
.submit-btn.loading {
    pointer-events: none;
    opacity: 0.8;
}

.submit-btn.loading::after {
    content: '';
    position: absolute;
    width: 20px;
    height: 20px;
    top: 50%;
    left: 50%;
    margin-left: -10px;
    margin-top: -10px;
    border: 2px solid #ffffff;
    border-radius: 50%;
    border-top-color: transparent;
    animation: spin 1s ease-in-out infinite;
}

@keyframes spin {
    to { transform: rotate(360deg); }
}

/* Three.js canvas styling */
#three-canvas {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    z-index: 0;
}
</style>

<div class="contact-page-wrapper">
    <div class="contact-container">
        <div class="contact-content">
           
            
            <form id="contact-form" class="modern-contact-form" method="post">
                <div class="form-row">
                    <div class="form-group">
                        <label for="first-name">First Name</label>
                        <input type="text" id="first-name" name="first_name" required>
                    </div>
                    <div class="form-group">
                        <label for="last-name">Last Name</label>
                        <input type="text" id="last-name" name="last_name" required>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="email">Email Address</label>
                    <input type="email" id="email" name="email" required>
                </div>
                
                <div class="form-group">
                    <label for="phone">Phone Number</label>
                    <input type="tel" id="phone" name="phone">
                </div>
                
                <div class="form-group">
                    <label for="subject">Subject</label>
                    <input type="text" id="subject" name="subject" required>
                </div>
                
                <div class="form-group">
                    <label for="message">Your Message</label>
                    <textarea id="message" name="message" rows="5" required></textarea>
                </div>
                
               
                
                <button type="submit" class="submit-btn">Send Message</button>
            </form>
        </div>
    </div>
</div>

<script type="importmap">
    {
        "imports": {
            "three": "https://cdn.jsdelivr.net/npm/three@0.161/build/three.module.js",
            "jsm/": "https://cdn.jsdelivr.net/npm/three@0.161/examples/jsm/"
        }
    }
</script>

<script type="module">
import * as THREE from "three";
import { OrbitControls } from "jsm/controls/OrbitControls.js";

const w = window.innerWidth;
const h = window.innerHeight;
const scene = new THREE.Scene();
const camera = new THREE.PerspectiveCamera(75, w / h, 1, 100);
camera.position.z = 5;
camera.position.y = -4;
const renderer = new THREE.WebGLRenderer({ antialias: true, alpha: true });
renderer.setSize(w, h);
renderer.domElement.id = 'three-canvas';
document.body.prepend(renderer.domElement);

const ctrls = new OrbitControls(camera, renderer.domElement);
ctrls.enableDamping = false;
ctrls.enableZoom = false; 
ctrls.enablePan = false;
ctrls.enableRotate = false;

function createTextTexture() {
    const canvas = document.createElement('canvas');
    const context = canvas.getContext('2d');
    canvas.width = 1024;
    canvas.height = 1024;
    context.imageSmoothingEnabled = true;
    context.textRenderingOptimization = 'optimizeQuality';

    context.fillStyle = 'black';
    context.fillRect(0, 0, canvas.width, canvas.height);
    context.fillStyle = '#e4e4e4d3';
    context.font = 'bold 55px Arial, sans-serif';
    context.textAlign = 'center';
    context.textBaseline = 'middle';

    context.textRenderingOptimization = 'optimizeQuality';

    const text = 'Escope';
    const textMetrics = context.measureText(text);
    const textWidth = textMetrics.width;
    const textHeight = 89;

    const horizontalSpacing = textWidth * .4;
    const verticalSpacing = textHeight * 2.3;

    for (let x = horizontalSpacing / 2; x < canvas.width; x += horizontalSpacing) {
        for (let y = verticalSpacing / 2; y < canvas.height; y += verticalSpacing) {
            context.save();
            context.translate(x, y);
            context.rotate(Math.PI / 2);
            context.fillText(text, 0, 0);
            context.restore();
        }
    }

    const texture = new THREE.CanvasTexture(canvas);
    texture.wrapS = THREE.RepeatWrapping;
    texture.wrapT = THREE.RepeatWrapping;
    texture.repeat.set(3, 3);
    texture.minFilter = THREE.LinearFilter;
    texture.magFilter = THREE.LinearFilter;
    texture.generateMipmaps = true;

    return texture;
}

const torusKnotGeo = new THREE.TorusGeometry(5, 3.8, 60, 100);
const textTexture = createTextTexture();
const torusKnotMat = new THREE.MeshStandardMaterial({
    map: textTexture
});
const torusKnot = new THREE.Mesh(torusKnotGeo, torusKnotMat);
torusKnot.rotation.x = Math.PI * 0.01;
torusKnot.position.set(0, 0, 0);
scene.add(torusKnot);

const spotLight = new THREE.SpotLight('#ffffff', 100);
spotLight.angle = Math.PI / 4.3;
spotLight.penumbra = .25;
spotLight.position.set(0, -0.2, 9);
scene.add(spotLight);

function animate(t = 0) {
    requestAnimationFrame(animate);
    torusKnot.rotation.z = t * 0.00009;
    const speed = 0.000009;
    textTexture.offset.y = - (t * speed) % 1;

    renderer.render(scene, camera);
    ctrls.update();
}

animate();

window.addEventListener('resize', () => {
    camera.aspect = window.innerWidth / window.innerHeight;
    camera.updateProjectionMatrix();
    renderer.setSize(window.innerWidth, window.innerHeight);
});
</script>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const contactForm = document.getElementById('contact-form');
    
    contactForm.addEventListener('submit', function(e) {
        e.preventDefault();
        
        // Basic validation
        const firstName = document.getElementById('first-name').value;
        const lastName = document.getElementById('last-name').value;
        const email = document.getElementById('email').value;
        const subject = document.getElementById('subject').value;
        const message = document.getElementById('message').value;
        
        if (!firstName || !lastName || !email || !subject || !message) {
            alert('Please fill in all required fields.');
            return;
        }
        
        // Email validation
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(email)) {
            alert('Please enter a valid email address.');
            return;
        }
        
        // In a real implementation, you would submit the form data to the server here
        // For this example, we'll just show a success message
        alert('Thank you for your message! We will get back to you soon.');
        contactForm.reset();
    });
});
</script>

<?php get_footer(); ?>